
/** !------------------------------------------------------------------------->
*
*  @file <BusConfiguration>.cpp
*
*  @creationDate 21 February 2025 15:05:45
*
*  @brief 
*
*  @author
*
*  @description
*
*  @copyright
*    Copyright 2025, dSPACE GmbH. All rights reserved.
*   
*!-------------------------------------------------------------------------->*/

/*----------------------------------------------------------------------------*/
/* INCLUDES */
/*----------------------------------------------------------------------------*/
#include "BMUtils.h"
#include "Bus_Configuration__1__10897.h"
#include "Bus_Configuration__1__10897_tx.h"

#include "C1_1/BusManager_Bus_Configuration__1__10897_1.h"
#include "StateManager_generated.h"
#include <string.h>
#include <Std_Types.h>
#include <DsCan.h>

/*----------------------------------------------------------------------------*/
/* DEFINES / ENUMS */
/*----------------------------------------------------------------------------*/
typedef uint32 PduIdType;

typedef enum CanIfCanIdType
{
    EXTENDED_CAN,
    STANDARD_CAN,
    EXTENDED_FD_CAN_BRS_OFF,
    STANDARD_FD_CAN_BRS_OFF,
    EXTENDED_FD_CAN_BRS_ON,
    STANDARD_FD_CAN_BRS_ON,
    J1939PDUF1,
    J1939PDUF2
} CanIfCanIdType;

typedef struct CanIfIndexRange
{
    sint32 MinIndex;
    sint32 MaxIndex;
} CanIfIndexRange;

typedef struct CanIfIndexMapping
{
    uint8 Count;
    const CanIfIndexRange(*Indices)[8];
} CanIfIndexMapping;

typedef struct CanIf_RxIndicationMetaDataType
{
    /* RxMonitor handle */
    void* CanIfReceivedHrh;
    /* Indicates whether the message item must be read from rx monitor */
    boolean IsInterruptContext;
    /* Index of the configuration element */
    PduIdType CanRxPduId;
    /* Map Message format and BRS flag to CanIf configurations Message format */
    CanIfCanIdType CanIdType;
} CanIf_RxIndicationMetaDataType;

typedef void (*CanIf_RxIndicationFunction)(void* pCbArg, const DsSCanMsg_Item* pMsgItem);

typedef struct
{
    CanIf_RxIndicationFunction funcRxIndication;
    PduIdType idxRxConfig;
    uint32 ecuRxId;
} CanEcuRxEntryType;

typedef struct
{
    CanIfCanIdType CanIdType;
    uint32 CanId;
    const CanEcuRxEntryType* func;
    int size;
} CanRxPduConfigEntryType;

typedef struct CanClusterConfig
{
    boolean                Initialized;
    uint8                  ControllerId;
    DsTCanBoard            CanBoard;
    DsTCanBoard_Address*   BoardRef;
    DsTCanCh               ChannelRef;
    DsTCanMsg              HrhRef;
    DsTCanMsg              HthRef;
} CanClusterConfig;

/*----------------------------------------------------------------------------*/
/* FORWARD DECLARATIONS OF STATIC FUNCTIONS */
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/* PROFILING SUPPORT                                                          */
/*----------------------------------------------------------------------------*/
#if defined(DS_PLATFORM_SMART) && (defined(CANIF_ENABLE_PROFILING) || defined(BUS_CFG_ENABLE_PROFILING) || defined(BUS_CFGTX_ENABLE_PROFILING))
#include "DsApplicationInterface.h"
static boolean doFirstStepInitialization = TRUE;
#if defined(CANIF_ENABLE_PROFILING)
UNUSED static DsTProfilerSource DsCanBoard_update_PROFILER_SRC;
UNUSED static DsTProfilerSource DsCanMsg_transmitItem_PROFILER_SRC;
UNUSED static DsTProfilerSource BusCfg_CAN_Rx_PROFILER_SRC;
#endif
#if defined(BUS_CFGTX_ENABLE_PROFILING)
UNUSED static DsTProfilerSource BusCfgTx_SetCreditCounter_PROFILER_SRC;
#endif
#if defined(DS_PLATFORM_SMART) &&  defined(BUS_CFG_ENABLE_PROFILING)
static DsTProfilerSource MainFunction_Com_PROFILER_SRC;
static DsTProfilerSource MainFunction_Flush_TxQueue_PROFILER_SRC;
#endif

#endif

static void ConvertMessageIdTypeToCanIfIdType(const DsSCanMsg_Item* pMsgItem, CanIfCanIdType* CanRxPduId);
static Std_ReturnType BinarySearchRxPduId(const uint32 CanId, PduIdType* CanRxPduId, const uint8 HrhID, const CanIfCanIdType CanIdType);
static void Bus_Configuration__1__10897_CANReceive(CanClusterConfig* clusterConfig);
static void Bus_Configuration__1__10897_CANOnStart(CanClusterConfig* clusterConfig);
static void Bus_Configuration__1__10897_CANOnStop(CanClusterConfig* clusterConfig);
static void Bus_Configuration__1__10897_CANInit(CanClusterConfig* clusterConfig);

/*----------------------------------------------------------------------------*/
/* GLOBAL VARIABLES */
/*----------------------------------------------------------------------------*/
#if defined(HOST_DEBUG)
EXTERN_C_DECL DsTCanBoard_Address* gp_CanIf_ChannelAddress[];
#endif
EXTERN_C_DECL void* Bus_Configuration__1__10897_CanIfHthIdSymRef[1];
void* Bus_Configuration__1__10897_CanIfHthIdSymRef[1];

/*----------------------------------------------------------------------------*/
/* GLOBAL FUNCTIONS */
/*----------------------------------------------------------------------------*/
EXTERN_C_DECL void BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_CanIf_RxIndication(void* pCbArg, const DsSCanMsg_Item* pMsgItem);
EXTERN_C_DECL void BM_BUS_CONFIGURATION__1__10897_1_PLANT_CanIf_RxIndication(void* pCbArg, const DsSCanMsg_Item* pMsgItem);

static CanClusterConfig g_clusterConfig[] = {
/* ThermalCAN */
    {
        FALSE,
        0,
        (DsTCanBoard) 0,
#if defined(HOST_DEBUG)
        gp_CanIf_ChannelAddress[0],
#else
        &BusManager_Bus_Configuration__1__10897_1_ThermalCAN_S,
#endif
        (DsTCanCh)  0,
        (DsTCanMsg) 0,
        (DsTCanMsg) 0
    },
};

static const CanIfIndexRange ga_CanIfIndexRanges[][8] = {
    /* EXTENDED_CAN, STANDARD_CAN, EXTENDED_FD_CAN_BRS_OFF, STANDARD_FD_CAN_BRS_OFF, EXTENDED_FD_CAN_BRS_ON, STANDARD_FD_CAN_BRS_ON, J1939PDUF1, J1939PDUF2 */
{/* ThermalCAN */    {-1, -1},     {0, 6},     {-1, -1},     {-1, -1},     {-1, -1},     {-1, -1},     {-1, -1},     {-1, -1}},
};
static const CanIfIndexMapping ga_CanIfIndexMapping = {1, ga_CanIfIndexRanges};

static const CanEcuRxEntryType func0_1_0[] =
{ /* 0 */
    { BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_CanIf_RxIndication, 0, 0 },
};
static const CanEcuRxEntryType func1_1_0[] =
{ /* 1 */
    { BM_BUS_CONFIGURATION__1__10897_1_PLANT_CanIf_RxIndication, 0, 1 },
};
static const CanEcuRxEntryType func2_1_0[] =
{ /* 2 */
    { BM_BUS_CONFIGURATION__1__10897_1_PLANT_CanIf_RxIndication, 1, 1 },
};
static const CanEcuRxEntryType func3_1_0[] =
{ /* 3 */
    { BM_BUS_CONFIGURATION__1__10897_1_PLANT_CanIf_RxIndication, 2, 1 },
};
static const CanEcuRxEntryType func4_1_0[] =
{ /* 4 */
    { BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_CanIf_RxIndication, 1, 0 },
};
static const CanEcuRxEntryType func5_1_0[] =
{ /* 5 */
    { BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_CanIf_RxIndication, 2, 0 },
};
static const CanEcuRxEntryType func6_1_0[] =
{ /* 6 */
    { BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_CanIf_RxIndication, 3, 0 },
};

static const CanRxPduConfigEntryType RxPduConfig[] =
{
/* ThermalCAN */
    { STANDARD_CAN /* CanIdType */, 0 /* 0x00000000 CAN Id */, func0_1_0, 1 }, /* 0 */
    { STANDARD_CAN /* CanIdType */, 1 /* 0x00000001 CAN Id */, func1_1_0, 1 }, /* 1 */
    { STANDARD_CAN /* CanIdType */, 2 /* 0x00000002 CAN Id */, func2_1_0, 1 }, /* 2 */
    { STANDARD_CAN /* CanIdType */, 3 /* 0x00000003 CAN Id */, func3_1_0, 1 }, /* 3 */
    { STANDARD_CAN /* CanIdType */, 4 /* 0x00000004 CAN Id */, func4_1_0, 1 }, /* 4 */
    { STANDARD_CAN /* CanIdType */, 5 /* 0x00000005 CAN Id */, func5_1_0, 1 }, /* 5 */
    { STANDARD_CAN /* CanIdType */, 6 /* 0x00000006 CAN Id */, func6_1_0, 1 }, /* 6 */
};

static void Bus_Configuration__1__10897_CANInit(CanClusterConfig* clusterConfig)
{
    if (clusterConfig->Initialized) return;

    DsCanBoard_create(&clusterConfig->CanBoard, *clusterConfig->BoardRef);
    DsCanCh_create(&clusterConfig->ChannelRef, clusterConfig->CanBoard, *clusterConfig->BoardRef);

    DsCanMsg_createRxMonitor(&clusterConfig->HrhRef, clusterConfig->ChannelRef);
    DsCanMsg_setQueueSize(clusterConfig->HrhRef, 100);
    DsCanMsg_apply(clusterConfig->HrhRef);
    DsCanMsg_start(clusterConfig->HrhRef);

    DsCanMsg_createTxQueue(&clusterConfig->HthRef, clusterConfig->ChannelRef);
    DsCanMsg_setQueueSize(clusterConfig->HthRef, 200);
    DsCanMsg_apply(clusterConfig->HthRef);
    DsCanMsg_start(clusterConfig->HthRef);
    Bus_Configuration__1__10897_CanIfHthIdSymRef[clusterConfig->ControllerId] = clusterConfig->HthRef;
    DsCanCh_start(clusterConfig->ChannelRef);

    clusterConfig->Initialized = TRUE;
}

static void ConvertMessageIdTypeToCanIfIdType(const DsSCanMsg_Item* pMsgItem, CanIfCanIdType* CanRxPduId)
{
    /* Map Message format and BRS flag to CanIf configurations Message format */
    switch (pMsgItem->Format)
    {
    case DSCAN_MSG_FORMAT_STD:
        *CanRxPduId = STANDARD_CAN;
        break;
    case DSCAN_MSG_FORMAT_EXT:
        *CanRxPduId = EXTENDED_CAN;
        break;
    case DSCAN_MSG_FORMAT_FD_STD:
        if (pMsgItem->Brs == DSCAN_MSG_BRS_ENABLED)
            *CanRxPduId = STANDARD_FD_CAN_BRS_ON;
        else
            *CanRxPduId = STANDARD_FD_CAN_BRS_OFF;
        break;
    case DSCAN_MSG_FORMAT_FD_EXT:
        if (pMsgItem->Brs == DSCAN_MSG_BRS_ENABLED)
            *CanRxPduId = EXTENDED_FD_CAN_BRS_ON;
        else
            *CanRxPduId = EXTENDED_FD_CAN_BRS_OFF;
        break;
    case DSCAN_MSG_FORMAT_BOTH:
    case DSCAN_MSG_FORMAT_FD_BOTH:
    case DSCAN_MSG_FORMAT_ALL:
    default:
        /* For unexpected values handle it as STANDARD_CAN */
        *CanRxPduId = STANDARD_CAN;
        break;
    }
}

static Std_ReturnType BinarySearchRxPduId(const uint32 CanId, PduIdType* CanRxPduId, const uint8 HrhID, const CanIfCanIdType CanIdType)
{
    /* Initialize return value to not ok */
    Std_ReturnType CanIfRet = E_NOT_OK;
    /* Variables for holding the index in binary search */
    sint32 MinInd = 0;
    sint32 MaxInd;
    sint32 MidInd;
    /* variable for holding the RxPduId */
    PduIdType RxPduId;
    const CanIfIndexMapping* indexMapping = &ga_CanIfIndexMapping;
    const CanIfIndexRange(*indices)[8] = indexMapping->Indices;

    if (HrhID >= indexMapping->Count)
        return E_NOT_OK;

    const CanIfIndexRange indexRange = indices[HrhID][(int)CanIdType];
    MinInd = indexRange.MinIndex;
    MaxInd = indexRange.MaxIndex;
    if (MinInd < 0 || MaxInd < 0)
        return E_NOT_OK;

    /* while end of searching for RxPdus is not reached */
    while ((MinInd < MaxInd) && (E_OK != CanIfRet))
    {
        /* Get the RxPduId in the middle of RxPduList */
        MidInd = MinInd + ((MaxInd - MinInd) / 2);
        RxPduId = (PduIdType)MidInd;
        {
            /*If only one CanID is expected per PduId */
            if (CanId < RxPduConfig[RxPduId].CanId)
            {
                /* update max index to (mid index) */
                MaxInd = MidInd;
            }
            /* if received CanId is greater than the corresponding RxPdu's CanId*/
            else if (CanId > RxPduConfig[RxPduId].CanId)
            {
                /* update min index to (mid index + 1) */
                MinInd = MidInd + 1;
            }
            else
            {
                *CanRxPduId = RxPduId;
                CanIfRet = E_OK;
            }
        }
    }

    /* if RxPduId corresponding to CanId was not found until (MaxInd = MinInd) */
    if (E_OK != CanIfRet)
    {
        /* If the CanId of the RxPduId corresponding to MinInd
        (which is now equal to MaxInd) is equal to required CanId */
        RxPduId = (PduIdType)MinInd;
        if (CanId == RxPduConfig[RxPduId].CanId)
        {
            /* Update the RxPduId */
            *CanRxPduId = RxPduId;
            CanIfRet = E_OK;
        }
    }
    return CanIfRet;
}



static void Bus_Configuration__1__10897_CANReceive(CanClusterConfig* clusterConfig)
{
    DsSCanMsg_Item msg_item;

    while(DsCanMsg_readRxItem(clusterConfig->HrhRef, &msg_item) == DSCAN_NO_ERROR)
    {
        if(msg_item.Status == DSCAN_MSG_NO_DATA)
        {
            break;
        }

        {
            uint32 CanId;
            PduIdType CanRxPduId = (PduIdType)-1;
            CanIfCanIdType CanIdType;
            Std_ReturnType CanIfRet;

            /* fix for DsCan */
            CanId = msg_item.Id & 0x1fffffff;

            /* Map Message format and BRS flag to CanIf configurations Message format */
            ConvertMessageIdTypeToCanIfIdType(&msg_item, &CanIdType);

            if ((CanIfRet = BinarySearchRxPduId(CanId, &CanRxPduId, clusterConfig->ControllerId, CanIdType)) == E_OK)
            {
                const CanRxPduConfigEntryType* pRxPduConfigEntry = &RxPduConfig[CanRxPduId];
                for (int i = 0; i < pRxPduConfigEntry->size; i++)
                {
                    const CanEcuRxEntryType* pEcuRxEntry = &pRxPduConfigEntry->func[i];
                    CanIf_RxIndicationMetaDataType CanIf_RxIndicationMetaData = {0, FALSE, pEcuRxEntry->idxRxConfig, CanIdType};
                    pEcuRxEntry->funcRxIndication((void*)&CanIf_RxIndicationMetaData, (const DsSCanMsg_Item*)&msg_item);
                }
            }
        }
    }
}

static void Bus_Configuration__1__10897_CANOnStart(CanClusterConfig* clusterConfig)
{
    if (clusterConfig->CanBoard != (DsTCanBoard)0)
    {
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
        dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanStart, 0, clusterConfig->ControllerId);
#endif
        DsCanBoard_update(clusterConfig->CanBoard);
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
        dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanEnd, 0, clusterConfig->ControllerId);
#endif
    }

    if (clusterConfig->HrhRef != (DsTCanMsg)0)
        DsCanMsg_start(clusterConfig->HrhRef);
}

static void Bus_Configuration__1__10897_CANOnStop(CanClusterConfig* clusterConfig)
{
    if (clusterConfig->HrhRef != (DsTCanMsg)0)
        DsCanMsg_stop(clusterConfig->HrhRef);

    if (clusterConfig->CanBoard != (DsTCanBoard)0)
    {
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
        dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanStart, 0, clusterConfig->ControllerId);
#endif
        DsCanBoard_update(clusterConfig->CanBoard);
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
        dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanEnd, 0, clusterConfig->ControllerId);
#endif
    }
}



/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_Receive
 *** 
 *** DESCRIPTION:
 ***     This function is called in Bus Configuration runnable function.
 ***
 *** PARAMETERS:
 ***     -/-
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void Bus_Configuration__1__10897_Receive(void)
{
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
    dsProfilerWriteEvent(BusCfg_CAN_Rx_PROFILER_SRC, UserTimespanStart, 0, g_clusterConfig[0].ControllerId);
#endif
    Bus_Configuration__1__10897_CANReceive(&g_clusterConfig[0]);
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
    dsProfilerWriteEvent(BusCfg_CAN_Rx_PROFILER_SRC, UserTimespanEnd, 0, g_clusterConfig[0].ControllerId);
#endif

}

/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_OnStart
 *** 
 *** DESCRIPTION:
 ***     This function is called in OnStart application phase.
 ***
 *** PARAMETERS:
 ***     -/-
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void Bus_Configuration__1__10897_OnStart(void)
{
    Bus_Configuration__1__10897_CANOnStart(&g_clusterConfig[0]);
    Bus_Configuration__1__10897_BusCfgTx_OnStart();
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_OnStop
 *** 
 *** DESCRIPTION:
 ***     This function is called in OnStop application phase.
 ***
 *** PARAMETERS:
 ***     -/-
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void Bus_Configuration__1__10897_OnStop(void)
{
    Bus_Configuration__1__10897_CANOnStop(&g_clusterConfig[0]);
    Bus_Configuration__1__10897_BusCfgTx_OnStop();
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_VariableInitialize
 *** 
 *** DESCRIPTION:
 ***    This function is called before the application starts and may be used to extend the
 ***    initialization phase.
 ***
 *** PARAMETERS:
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void Bus_Configuration__1__10897_VariableInitialize(int controllerId)
{
    POSSIBLE_UNREFERENCED_PARAMETER(controllerId);
    StateManager_SetState(1, 0);
    if (controllerId >= 0 && controllerId < 1)
    {
        Bus_Configuration__1__10897_CANInit(&g_clusterConfig[controllerId]);
    }
#if defined(DS_PLATFORM_SMART) && (defined(CANIF_ENABLE_PROFILING) || defined(BUS_CFG_ENABLE_PROFILING)|| defined(BUS_CFGTX_ENABLE_PROFILING))
    if (doFirstStepInitialization)
    {
#if defined(CANIF_ENABLE_PROFILING)
        DsCanBoard_update_PROFILER_SRC = dsProfilerCreateSource("Bus_Configuration__1__10897""_DsCanBoard_update");
        DsCanMsg_transmitItem_PROFILER_SRC = dsProfilerCreateSource("Bus_Configuration__1__10897""_DsCanMsg_transmitItem");
        BusCfg_CAN_Rx_PROFILER_SRC = dsProfilerCreateSource("Bus_Configuration__1__10897""_CAN_RX");
#endif
    PROFILER_CREATE_SET_CREDIT_COUNTER("Bus_Configuration__1__10897_BusCfgTx_SetCreditCounter");
#if defined(DS_PLATFORM_SMART) && defined(BUS_CFG_ENABLE_PROFILING)
		MainFunction_Com_PROFILER_SRC = dsProfilerCreateSource("Bus_Configuration__1__10897_Com_MainFunction");
		MainFunction_Flush_TxQueue_PROFILER_SRC = dsProfilerCreateSource("Bus_Configuration__1__10897_Flush_TxQueue_MainFunction");
#endif

        doFirstStepInitialization = FALSE;
    }
#endif
    Bus_Configuration__1__10897_BusCfgTx_VariableInitialize();
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_AdjustCreditCounter
 *** 
 *** DESCRIPTION:
 ***    This function is called to adjust the credit counter for specific CAN controller.
 ***    If at begin of a sample step still some messages are in TX queue, the credit counter 
 ***    will be decreased by number of outstanding TX messages.
 ***
 *** PARAMETERS:
 ***    int controllerId 
 ***    int stateManagerIndex - stateManagerIndex of credit counter
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void Bus_Configuration__1__10897_AdjustCreditCounter(int controllerId, int stateManagerIndex)
{
    POSSIBLE_UNREFERENCED_PARAMETER(controllerId);
    POSSIBLE_UNREFERENCED_PARAMETER(stateManagerIndex);

    if (stateManagerIndex < 0)
        return;

    uint32 QueueCount;
    DsTError DsCanRet;

    DsTCanMsg TxQueueHandle = (DsTCanMsg) g_clusterConfig[controllerId].HthRef;
    DsCanRet = DsCanMsg_getTxQueueCount(TxQueueHandle, (UInt32*)&QueueCount);
    if ((DsCanRet == DSCAN_NO_ERROR) && (QueueCount > 0))
    {
        uint32 currentCounter = StateManager_GetState((uint32)stateManagerIndex);
        if (currentCounter > QueueCount)
        {
            StateManager_SetState((uint32)stateManagerIndex, (uint8)(currentCounter - QueueCount));
        }
        else 
        {
            StateManager_SetState((uint32)stateManagerIndex, 0);
        }
        PROFILER_TIMESTAMP_SET_CREDIT_COUNTER(stateManagerIndex, StateManager_GetState((uint32)stateManagerIndex));
    }
}


/**************************************************************************************************\
 *** FUNCTION:
 ***    DoPossibleCreditCounterInitialization
 *** 
 *** DESCRIPTION:
 ***    This function is called at start of bus configuration MainFunction to check if 
 ***    reset of credit counter is necessary. Only one bus configuration MainFunction will
 ***    do this, controlled by a StateManager Counter State.
 ***
 *** PARAMETERS:
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
UNUSED static void DoPossibleCreditCounterInitialization(void)
{
    if (0 == StateManager_GetState(1))
    {
        /* Reset 'all' credit counters to default values */
        StateManager_ResetCounterState(0);
        PROFILER_TIMESTAMP_SET_CREDIT_COUNTER(0, StateManager_GetState(0));

        /* Reset counter for next simulation step */
        StateManager_ResetCounterState(1);
    }
    /* Decrement counter to indicate check has been done */
    StateManager_DecrementCounterState(1);
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     Bus_Configuration__1__10897_BusCfg_MainFunction_<..>
 *** 
 *** DESCRIPTION:
 ***     Bus Configuration MainFunction
 *** 
 *** PARAMETERS:
 ***     -/-
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
#ifndef HOST_DEBUG
void Bus_Configuration__1__BusCfg_MainFunction_10896_10894(void)
{
    DoPossibleCreditCounterInitialization();
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
    dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanStart, 0u, 0);
#endif
    DsCanBoard_update(g_clusterConfig[0].CanBoard);
#if defined(DS_PLATFORM_SMART) && defined(CANIF_ENABLE_PROFILING)
    dsProfilerWriteEvent(DsCanBoard_update_PROFILER_SRC, UserTimespanEnd, 0u, 0);
#endif

    Bus_Configuration__1__10897_AdjustCreditCounter(0, 0);
    Bus_Configuration__1__10897_Receive();
#if defined(DS_PLATFORM_SMART) && defined(BUS_CFG_ENABLE_PROFILING)
    dsProfilerWriteEvent(MainFunction_Com_PROFILER_SRC, UserTimespanStart, 0, 0);
#endif
    BM_BUS_CONFIGURATION__1__10897_1_CONTROLLER_Com_MainFunction();
    BM_BUS_CONFIGURATION__1__10897_1_PLANT_Com_MainFunction();
#if defined(DS_PLATFORM_SMART) && defined(BUS_CFG_ENABLE_PROFILING)
    dsProfilerWriteEvent(MainFunction_Com_PROFILER_SRC, UserTimespanEnd, 0, 0);
#endif
    #if defined(DS_PLATFORM_SMART) && defined(BUS_CFG_ENABLE_PROFILING)
    dsProfilerWriteEvent(MainFunction_Flush_TxQueue_PROFILER_SRC, UserTimespanStart, 0, 0);
#endif
if (StateManager_GetState(3) == BUSCFG_STATE_ENABLED) Bus_Configuration__1__10897_BusCfgTx_TriggerPendingPdus();
#if defined(DS_PLATFORM_SMART) && defined(BUS_CFG_ENABLE_PROFILING)
    dsProfilerWriteEvent(MainFunction_Flush_TxQueue_PROFILER_SRC, UserTimespanEnd, 0, 0);
#endif

}
#endif


